<?php

namespace App\Http\Controllers\Admin;

use Auth;
use App\User;
use App\Language;
use App\Genres;
use App\ActorDirector;
use App\Http\Requests;
use Illuminate\Http\Request;
use Session;
use Intervention\Image\Facades\Image;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;
use Exception;

class ImportImdbController extends MainAdminController
{
    private $tmdbApiKey;
    private $defaultLanguage;
    private $baseUrl = 'https://api.themoviedb.org/3';
    
    public function __construct()
    {
        $this->middleware('auth');
        $this->tmdbApiKey = getcong('tmdb_api_key');
        $this->defaultLanguage = set_tmdb_language();
    }

    /**
     * Find and import movie data from TMDB
     */
    public function find_imdb_movie(Request $request)
    {
        try {
            $movieId = $request->get('id');
            
            if (!$movieId) {
                return response()->json([
                    'imdb_status' => 'fail',
                    'message' => 'Movie ID is required'
                ]);
            }

            // Validate API key
            if (!$this->tmdbApiKey) {
                return response()->json([
                    'imdb_status' => 'fail',
                    'message' => 'TMDB API key not configured'
                ]);
            }

            // Fetch movie data with videos and credits
            $movieData = $this->fetchMovieData($movieId);
            
            if (!$movieData) {
                return response()->json([
                    'imdb_status' => 'fail',
                    'message' => 'Movie not found or API error'
                ]);
            }

            // Process and format the response
            $response = $this->processMovieData($movieData);
            
            return response()->json($response);

        } catch (Exception $e) {
            Log::error('TMDB Import Error: ' . $e->getMessage());
            
            return response()->json([
                'imdb_status' => 'fail',
                'message' => 'An error occurred while fetching movie data'
            ]);
        }
    }

    /**
     * Fetch movie data from TMDB API
     */
    private function fetchMovieData($movieId)
    {
        $url = "{$this->baseUrl}/movie/{$movieId}?language={$this->defaultLanguage}&append_to_response=videos,credits&api_key={$this->tmdbApiKey}";
        
        $curl = curl_init();
        
        curl_setopt_array($curl, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "GET",
            CURLOPT_HTTPHEADER => [
                'Accept: application/json',
                'User-Agent: MovieApp/1.0'
            ],
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_FOLLOWLOCATION => true
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        $error = curl_error($curl);
        
        curl_close($curl);

        if ($error) {
            Log::error("cURL Error: " . $error);
            return null;
        }

        if ($httpCode !== 200) {
            Log::error("HTTP Error: " . $httpCode);
            return null;
        }

        $result = json_decode($response);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            Log::error("JSON Decode Error: " . json_last_error_msg());
            return null;
        }

        return $result;
    }

    /**
     * Process movie data and return formatted response
     */
    private function processMovieData($movieData)
    {
        $response = [
            'imdb_status' => 'success',
            'imdbid' => $movieData->imdb_id ?? '',
            'imdb_rating' => isset($movieData->vote_average) ? round($movieData->vote_average, 1) : 0,
            'imdb_votes' => $movieData->vote_count ?? 0,
            'title' => $movieData->title ?? '',
            'runtime' => $this->formatRuntime($movieData->runtime ?? 0),
            'released' => isset($movieData->release_date) ? date('m/d/Y', strtotime($movieData->release_date)) : '',
            'plot' => $movieData->overview ?? '',
        ];

        // Process language
        $response['language'] = $this->processLanguage($movieData);

        // Process genres
        $response['genre'] = $this->processGenres($movieData);

        // Process cast
        $response['actors'] = $this->processActors($movieData);

        // Process directors
        $response['director'] = $this->processDirectors($movieData);

        // Process images
        $this->processImages($movieData, $response);

        // Process trailer
        $response['trailer_url'] = $this->processTrailer($movieData);

        return $response;
    }

    /**
     * Format runtime from minutes to hours and minutes
     */
    private function formatRuntime($minutes)
    {
        if (!$minutes) return '';
        
        $hours = floor($minutes / 60);
        $mins = $minutes - ($hours * 60);
        
        return $hours . "h " . $mins . "m";
    }

    /**
     * Process movie language
     */
    private function processLanguage($movieData)
    {
        if (!isset($movieData->spoken_languages) || empty($movieData->spoken_languages)) {
            return null;
        }

        $langName = $movieData->spoken_languages[0]->english_name ?? '';
        return $langName ? Language::getLanguageID($langName) : null;
    }

    /**
     * Process movie genres
     */
    private function processGenres($movieData)
    {
        $genres = [];
        
        if (isset($movieData->genres) && is_array($movieData->genres)) {
            foreach ($movieData->genres as $genre) {
                if (isset($genre->name)) {
                    $genreId = Genres::getGenresID($genre->name);
                    if ($genreId) {
                        $genres[] = $genreId;
                    }
                }
            }
        }
        
        return $genres;
    }

    /**
     * Process movie actors
     */
    private function processActors($movieData)
    {
        $actorOptions = [];
        
        if (!isset($movieData->credits->cast) || !is_array($movieData->credits->cast)) {
            return $actorOptions;
        }

        $castMembers = array_slice($movieData->credits->cast, 0, 10); // Limit to first 10
        
        foreach ($castMembers as $castMember) {
            try {
                $actorId = $this->processActorDirector($castMember, 'actor');
                if ($actorId) {
                    $actorName = $castMember->original_name ?? $castMember->name ?? '';
                    $actorOptions[] = "<option value='{$actorId}' selected>{$actorName}</option>";
                }
            } catch (Exception $e) {
                Log::warning("Error processing actor: " . $e->getMessage());
                continue;
            }
        }
        
        return $actorOptions;
    }

    /**
     * Process movie directors
     */
    private function processDirectors($movieData)
    {
        $directorOptions = [];
        
        if (!isset($movieData->credits->crew) || !is_array($movieData->credits->crew)) {
            return $directorOptions;
        }

        foreach ($movieData->credits->crew as $crewMember) {
            if (isset($crewMember->job) && $crewMember->job === 'Director') {
                try {
                    $directorId = $this->processActorDirector($crewMember, 'director');
                    if ($directorId) {
                        $directorName = $crewMember->name ?? '';
                        $directorOptions[] = "<option value='{$directorId}' selected>{$directorName}</option>";
                    }
                } catch (Exception $e) {
                    Log::warning("Error processing director: " . $e->getMessage());
                    continue;
                }
            }
        }
        
        return $directorOptions;
    }

    /**
     * Process actor or director and save to database if not exists
     */
    private function processActorDirector($person, $type)
    {
        $name = $person->original_name ?? $person->name ?? '';
        $tmdbId = $person->id ?? null;
        
        if (!$name || !$tmdbId) {
            return null;
        }

        // Check if actor/director already exists
        $existingPerson = ActorDirector::where('ad_name', $name)
                                     ->where('ad_type', $type)
                                     ->first();

        if ($existingPerson) {
            return $existingPerson->id;
        }

        // Fetch detailed person information
        $personDetails = $this->fetchPersonDetails($tmdbId);
        
        if (!$personDetails) {
            return null;
        }

        // Create new actor/director record
        $actorDirector = new ActorDirector;
        $actorDirector->ad_type = $type;
        $actorDirector->ad_name = addslashes($name);
        $actorDirector->ad_bio = addslashes($personDetails->biography ?? '');
        $actorDirector->ad_birthdate = $personDetails->birthday ? strtotime($personDetails->birthday) : null;
        $actorDirector->ad_place_of_birth = addslashes($personDetails->place_of_birth ?? '');
        $actorDirector->ad_tmdb_id = $tmdbId;
        $actorDirector->ad_slug = Str::slug($name, '-', null);

        // Download and save profile image
        if (isset($person->profile_path) && $person->profile_path) {
            $imagePath = $this->downloadPersonImage($person->profile_path);
            if ($imagePath) {
                $actorDirector->ad_image = $imagePath;
            }
        }

        $actorDirector->save();
        
        return $actorDirector->id;
    }

    /**
     * Fetch person details from TMDB
     */
    private function fetchPersonDetails($personId)
    {
        $url = "{$this->baseUrl}/person/{$personId}?language={$this->defaultLanguage}&api_key={$this->tmdbApiKey}";
        
        $curl = curl_init();
        
        curl_setopt_array($curl, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 15,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "GET",
            CURLOPT_HTTPHEADER => [
                'Accept: application/json',
                'User-Agent: MovieApp/1.0'
            ],
            CURLOPT_SSL_VERIFYPEER => false
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        if ($httpCode === 200) {
            return json_decode($response);
        }

        return null;
    }

    /**
     * Download person profile image
     */
    private function downloadPersonImage($profilePath)
    {
        try {
            $imageUrl = 'https://image.tmdb.org/t/p/w300' . $profilePath;
            $fileName = basename(parse_url($imageUrl, PHP_URL_PATH));
            $savePath = public_path('/upload/images/' . $fileName);
            
            if (grab_image($imageUrl, $savePath)) {
                return 'upload/images/' . $fileName;
            }
        } catch (Exception $e) {
            Log::warning("Error downloading person image: " . $e->getMessage());
        }
        
        return null;
    }

    /**
     * Process movie images (poster and backdrop)
     */
    private function processImages($movieData, &$response)
    {
        // Process poster
        if (isset($movieData->poster_path) && $movieData->poster_path) {
            $posterPath = 'https://image.tmdb.org/t/p/w300' . $movieData->poster_path;
            $response['thumbnail'] = $posterPath;
            $response['thumbnail_name'] = basename(parse_url($posterPath, PHP_URL_PATH));
        }

        // Process backdrop
        if (isset($movieData->backdrop_path) && $movieData->backdrop_path) {
            $backdropPath = 'https://image.tmdb.org/t/p/w780' . $movieData->backdrop_path;
            $response['poster'] = $backdropPath;
            $response['poster_name'] = basename(parse_url($backdropPath, PHP_URL_PATH));
        }
    }

    /**
     * Process movie trailer
     */
    private function processTrailer($movieData)
    {
        if (!isset($movieData->videos->results) || !is_array($movieData->videos->results)) {
            return '';
        }

        foreach ($movieData->videos->results as $video) {
            if (isset($video->type) && $video->type === 'Trailer' && 
                isset($video->site) && $video->site === 'YouTube' && 
                isset($video->key)) {
                return 'https://www.youtube.com/watch?v=' . $video->key;
            }
        }

        return '';
    }

    /**
     * Alternative method to find movie by IMDb ID
     */
    public function find_by_imdb_id(Request $request)
    {
        try {
            $imdbId = $request->get('imdb_id');
            
            if (!$imdbId) {
                return response()->json([
                    'status' => 'fail',
                    'message' => 'IMDb ID is required'
                ]);
            }

            // Find movie by IMDb ID
            $url = "{$this->baseUrl}/find/{$imdbId}?external_source=imdb_id&api_key={$this->tmdbApiKey}";
            
            $curl = curl_init();
            curl_setopt_array($curl, [
                CURLOPT_URL => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => 30,
                CURLOPT_HTTPHEADER => ['Accept: application/json'],
                CURLOPT_SSL_VERIFYPEER => false
            ]);

            $response = curl_exec($curl);
            $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            curl_close($curl);

            if ($httpCode !== 200) {
                return response()->json([
                    'status' => 'fail',
                    'message' => 'Failed to fetch movie data'
                ]);
            }

            $result = json_decode($response);
            
            if (empty($result->movie_results)) {
                return response()->json([
                    'status' => 'fail',
                    'message' => 'No movie found with this IMDb ID'
                ]);
            }

            $movieId = $result->movie_results[0]->id;
            
            // Now fetch complete movie data
            $request->merge(['id' => $movieId]);
            return $this->find_imdb_movie($request);

        } catch (Exception $e) {
            Log::error('IMDb ID Search Error: ' . $e->getMessage());
            
            return response()->json([
                'status' => 'fail',
                'message' => 'An error occurred while searching for the movie'
            ]);
        }
    }
}